/*=============================================================================
	TcpNetDriver.cpp: Unreal TCP/IP driver.
	Copyright 1997-1999 Epic Games, Inc. All Rights Reserved.

Revision history:
	* Created by Tim Sweeney.

Notes:
	* See \msdev\vc98\include\winsock.h and \msdev\vc98\include\winsock2.h 
	  for Winsock WSAE* errors returned by Windows Sockets.
=============================================================================*/

#include "UnIpDrv.h"
#include "UnTcpNetDriver.h"

/*-----------------------------------------------------------------------------
	Declarations.
-----------------------------------------------------------------------------*/

/* yucky legacy hack. --ryan. */
#ifdef __linux__
#  ifndef MSG_ERRQUEUE
#    define MSG_ERRQUEUE 0x2000
#  endif
#endif

// Size of a UDP header.
#define IP_HEADER_SIZE     (20)
#define UDP_HEADER_SIZE    (IP_HEADER_SIZE+8)
#define SLIP_HEADER_SIZE   (UDP_HEADER_SIZE+4)
#define WINSOCK_MAX_PACKET (512)
#define NETWORK_MAX_PACKET (576)

// Variables.
UBOOL GIpDrvInitialized;

/*-----------------------------------------------------------------------------
	UTcpipConnection.
-----------------------------------------------------------------------------*/

//
// Windows socket class.
//
// Constructors and destructors.
UTcpipConnection::UTcpipConnection( SOCKET InSocket, UNetDriver* InDriver, FIpAddr InRemoteAddr, EConnectionState InState, UBOOL InOpenedLocally, const FURL& InURL )
:	UNetConnection	( InDriver, InURL )
,	Socket			( InSocket )
,	RemoteAddr		( InRemoteAddr )
,	OpenedLocally	( InOpenedLocally )
{
	guard(UTcpipConnection::UTcpipConnection);

	// Init the connection.
	State                 = InState;
	MaxPacket			  = WINSOCK_MAX_PACKET;
	PacketOverhead		  = SLIP_HEADER_SIZE;
	OpenedTime			  = GCurrentTime;
	InitOut();

	// In connecting, figure out IP address.
	if( InOpenedLocally )
	{
		FIpAddr Addr{ *InURL.Host, 0 };
		if (Addr.Family == AF_UNSPEC) {
            // Create thread to resolve the address.
            ResolveInfo = new FResolveInfo(*InURL.Host);
		} else {
			RemoteAddr = Addr;
		}
	}
	unguard;
}

void UTcpipConnection::LowLevelSend( void* Data, INT Count )
{
	guard(UTcpipConnection::LowLevelSend);
	if( ResolveInfo )
	{
		// If destination address isn't resolved yet, send nowhere.
		if( !ResolveInfo->Resolved() )
		{
			// Host name still resolving.
			return;
		}
		else if( ResolveInfo->GetError() )
		{
			// Host name resolution just now failed.
			debugf( NAME_Log, TEXT("%s"), ResolveInfo->GetError() );
			Driver->ServerConnection->State = USOCK_Closed;
			delete ResolveInfo;
			ResolveInfo = NULL;
			return;
		}
		else
		{
			// Host name resolution just now succeeded.
			RemoteAddr = ResolveInfo->Addr;
			debugf( TEXT("Resolved %s (%s)"), ResolveInfo->GetHostName(), *IpString(ResolveInfo->GetAddr()) );
			delete ResolveInfo;
			ResolveInfo = NULL;
		}
	}
		// Send to remote.
	clock(Driver->SendCycles);
	sendto( Socket, (char *)Data, Count, 0, (sockaddr*)&RemoteAddr, sizeof(RemoteAddr) );
	unclock(Driver->SendCycles);
	unguard;
}

FString UTcpipConnection::LowLevelGetRemoteAddress()
{
	guard(UTcpipConnection::LowLevelGetRemoteAddress);
	return RemoteAddr.GetString(true);
	unguard;
}

FString UTcpipConnection::LowLevelDescribe()
{
	guard(UTcpipConnection::LowLevelDescribe);
	return FString::Printf
	(
		TEXT("%s %s state: %s"),
		*URL.Host,
		*RemoteAddr.GetString(true),
			State==USOCK_Pending	?	TEXT("Pending")
		:	State==USOCK_Open		?	TEXT("Open")
		:	State==USOCK_Closed		?	TEXT("Closed")
		:								TEXT("Invalid")
	);
	unguard;
}

IMPLEMENT_CLASS(UTcpipConnection);

/*-----------------------------------------------------------------------------
	UTcpNetDriver.
-----------------------------------------------------------------------------*/

//
// Windows sockets network driver.
//
UBOOL UTcpNetDriver::InitConnect( FNetworkNotify* InNotify, FURL& ConnectURL, FString& Error )
{
	guard(UTcpNetDriver::InitConnect);
	if( !Super::InitConnect( InNotify, ConnectURL, Error ) )
		return 0;
	if( !InitBase( 1, InNotify, ConnectURL, Error ) )
		return 0;

	// Connect to remote.
	FStringOutputDevice Out;
	auto TempAddr = getlocalbindaddr(Out);
	TempAddr.Port = ConnectURL.Port;

	if (TempAddr.Family == AF_UNSPEC)
		Error = Out;

	// Create new connection.
	ServerConnection = new UTcpipConnection( Socket, this, TempAddr, USOCK_Pending, 1, ConnectURL );
	debugf( NAME_DevNet, TEXT("Game client on port %i, rate %i"), LocalAddr.Port, ServerConnection->CurrentNetSpeed );

	// Create channel zero.
	GetServerConnection()->CreateChannel( CHTYPE_Control, 1, 0 );

	return 1;
	unguard;
}

UBOOL UTcpNetDriver::InitListen( FNetworkNotify* InNotify, FURL& LocalURL, FString& Error )
{
	guard(UTcpNetDriver::InitListen);
	if( !Super::InitListen( InNotify, LocalURL, Error ) )
		return 0;
	if( !InitBase( 0, InNotify, LocalURL, Error ) )
		return 0;

	// Update result URL.
	LocalURL.Host = LocalAddr.GetString(false);
	LocalURL.Port = LocalAddr.Port;
	debugf( NAME_DevNet, TEXT("TcpNetDriver on port %i"), LocalURL.Port );

	return 1;
	unguard;
}

void UTcpNetDriver::TickDispatch( FLOAT DeltaTime )
{
	guard(UTcpNetDriver::TickDispatch);
	Super::TickDispatch( DeltaTime );

	DWORD TCPNetDriverTime = 0.f;
	clock(TCPNetDriverTime);
	// Process all incoming packets.
	BYTE Data[NETWORK_MAX_PACKET];
	sockaddr_storage FromAddr{};
	for( ; ; )
	{
		// Get data, if any.
		clock(RecvCycles);
		socklen_t FromAddrLen = sizeof(FromAddr);
		INT Size = recvfrom( Socket, (char*)Data, sizeof(Data), 0, (sockaddr*)&FromAddr, &FromAddrLen);
		unclock(RecvCycles);
		// Handle result.
		if( Size==SOCKET_ERROR )
		{
			INT Error = WSAGetLastError();
			if( Error == WSAEWOULDBLOCK )
			{
				// No data
				break;
			}
			else
			{
                #ifdef __linux__
                    // determine IP address where problem originated. --ryan.
                    recvfrom(Socket, NULL, 0, MSG_ERRQUEUE, (sockaddr*)&FromAddr, GCC_OPT_INT_CAST &FromSize );
                #endif
                
				if( Error != UDP_ERR_PORT_UNREACH )
				{
					static UBOOL FirstError=1;
					if( FirstError )
						debugf( TEXT("UDP recvfrom error: %i from %s"), WSAGetLastError(), *IpString((sockaddr*)&FromAddr, FromAddrLen, true));
					FirstError = 0;
					break;
				}
			}
		}
		// Figure out which socket the received data came from.
		UTcpipConnection* Connection = NULL;
		if (GetServerConnection()) {
			sockaddr_storage RemoteAddr;
			socklen_t RemoteAddrLen;
			GetServerConnection()->RemoteAddr.GetSockAddr(&RemoteAddr, &RemoteAddrLen);
			if (IpMatches((sockaddr*)&RemoteAddr, (sockaddr*)&FromAddr))
				Connection = GetServerConnection();
		}

		for (INT i = 0; i < ClientConnections.Num() && !Connection; i++) {
            sockaddr_storage RemoteAddr;
            socklen_t RemoteAddrLen;
			((UTcpipConnection*)ClientConnections(i))->RemoteAddr.GetSockAddr(&RemoteAddr, &RemoteAddrLen);
			if (IpMatches((sockaddr*)&RemoteAddr, (sockaddr*)&FromAddr))
				Connection = (UTcpipConnection*)ClientConnections(i);
		}

		if( Size==SOCKET_ERROR )
		{
			if( Connection )
			{
				if( Connection != GetServerConnection() )
				{
					// We received an ICMP port unreachable from the client, meaning the client is no longer running the game
					// (or someone is trying to perform a DoS attack on the client)

					// rcg08182002 Some buggy firewalls get occasional ICMP port
					// unreachable messages from legitimate players. Still, this code
					// will drop them unceremoniously, so there's an option in the .INI
					// file for servers with such flakey connections to let these
					// players slide...which means if the client's game crashes, they
					// might get flooded to some degree with packets until they timeout.
					// Either way, this should close up the usual DoS attacks.
					if ((Connection->State != USOCK_Open) || (!AllowPlayerPortUnreach))
					{
						if (LogPortUnreach)
							debugf( TEXT("Received ICMP port unreachable from client %s.  Disconnecting."), *IpString((sockaddr*)&FromAddr, FromAddrLen, true));
						delete Connection;
					}
				}
			}
			else
			{
				if (LogPortUnreach)
					debugf( TEXT("Received ICMP port unreachable from %s.  No matching connection found."), *IpString((sockaddr*)&FromAddr, FromAddrLen, true));
			}
		}
		else
		{
			// Go through existing connection list and see how many connections for this IP we have opened in the last minute.
			if( MaxConnPerIPPerMinute == 0 )
				MaxConnPerIPPerMinute = 5;
			INT SameIPCount=0;
			INT SameIPRangeCount = 0;
			if ( LimitConnPerIPRangePerMinute && (ClientConnections.Num() > 32) )
			{ 
			    for( INT i=0; i<ClientConnections.Num() && !Connection; i++ )
			    {
                    sockaddr_storage RemoteAddr;
                    socklen_t RemoteAddrLen;
                    ((UTcpipConnection*)ClientConnections(i))->RemoteAddr.GetSockAddr(&RemoteAddr, &RemoteAddrLen);

				    // prevent DOS attacks on subnets
				    // count connects from same subnet.  Don't allow more than 32 per minute, since that's how many players we support on a server
				    if( IpAddressRangeMatches((sockaddr*)&RemoteAddr, (sockaddr*)&FromAddr) && ((UTcpipConnection*)ClientConnections(i))->OpenedTime > (GCurrentTime - 60.0) )
				    {
					    SameIPRangeCount++;
					    if( IpAddressMatches((sockaddr*)&RemoteAddr, (sockaddr*)&FromAddr) )
						    SameIPCount++;
				    }
			    }
			}
			else
			{
				for (INT i = 0; i < ClientConnections.Num() && !Connection; i++) {
                    sockaddr_storage RemoteAddr;
                    socklen_t RemoteAddrLen;
                    ((UTcpipConnection*)ClientConnections(i))->RemoteAddr.GetSockAddr(&RemoteAddr, &RemoteAddrLen);
					if (IpAddressMatches((sockaddr*)&RemoteAddr, (sockaddr*)&FromAddr) && ((UTcpipConnection*)ClientConnections(i))->OpenedTime > (GCurrentTime - 60.0))
						SameIPCount++;
				}
			}

			if( (SameIPCount < MaxConnPerIPPerMinute) && (SameIPRangeCount < 32) )
			{
				// If we didn't find a client connection, maybe create a new one.
				if( !Connection && Notify->NotifyAcceptingConnection()==ACCEPTC_Accept )
				{
					Connection = new UTcpipConnection( Socket, this, FIpAddr((sockaddr*)&FromAddr), USOCK_Open, 0, FURL() );
					Connection->URL.Host = IpString((sockaddr*)&FromAddr, FromAddrLen);
					Notify->NotifyAcceptedConnection( Connection );
					ClientConnections.AddItem( Connection );
				}

	unclock(TCPNetDriverTime);
				// Send the packet to the connection for processing.
				if( Connection )
					Connection->ReceivedRawPacket( Data, Size );
	clock(TCPNetDriverTime);
			}
			else
			if( LogMaxConnPerIPPerMin )
				debugf( TEXT("More than %d logins per minute for %s.  Ignoring."), MaxConnPerIPPerMinute, *IpString((sockaddr*)&FromAddr, FromAddrLen) );
		}
	}
	unclock(TCPNetDriverTime);
	//if ( TCPNetDriverTime * GSecondsPerCycle * 1000.f > 1.f )
	//		debugf(TEXT("TCPNetDriver %01.4f"), TCPNetDriverTime * GSecondsPerCycle * 1000.f);

	unguard;
}

FString UTcpNetDriver::LowLevelGetNetworkNumber()
{
	guard(UTcpNetDriver::LowLevelGetNetworkNumber);
	return LocalAddr.GetString(false);
	unguard;
}

void UTcpNetDriver::LowLevelDestroy()
{
	guard(UTcpNetDriver::LowLevelDestroy);

	// Close the socket.
	if( Socket )
	{
		if( closesocket(Socket) )
			debugf( NAME_Exit, TEXT("TcpNetDriver: Socket close error (%i)"), WSAGetLastError() );
		Socket=NULL;
		debugf( NAME_Exit, TEXT("Socket shut down") );
	}

	unguard;
}

// UTcpNetDriver interface.
UBOOL UTcpNetDriver::InitBase( UBOOL Connect, FNetworkNotify* InNotify, FURL& URL, FString& Error )
{
	guard(UTcpNetDriver::UTcpNetDriver);

	// Init WSA.
	if( !InitSockets( Error ) )
		return 0;

	// Create UDP socket and enable broadcasting.
	Socket = socket( AF_INET, SOCK_DGRAM, IPPROTO_UDP );
	if( Socket == INVALID_SOCKET )
	{
		Socket = 0;
		Error = FString::Printf( TEXT("TcpNetDriver: socket failed (%i)"), SocketError() );
		return 0;
	}
	UBOOL TrueBuffer=1;
	if( setsockopt( Socket, SOL_SOCKET, SO_BROADCAST, (char*)&TrueBuffer, sizeof(TrueBuffer) ) )
	{
		Error = FString::Printf( TEXT("%s: setsockopt SO_BROADCAST failed (%i)"), SOCKET_API, SocketError() );
		return 0;
	}
	if( !SetSocketReuseAddr( Socket ) )
		debugf(TEXT("setsockopt with SO_REUSEADDR failed"));

	if( !SetSocketRecvErr( Socket ) )
		debugf(TEXT("setsockopt with IP_RECVERR failed"));

    // Increase socket queue size, because we are polling rather than threading
	// and thus we rely on Windows Sockets to buffer a lot of data on the server.
	INT RecvSize = Connect ? 0x8000 : 0x20000, SizeSize=sizeof(RecvSize);
	INT SendSize = Connect ? 0x8000 : 0x20000;
	setsockopt( Socket, SOL_SOCKET, SO_RCVBUF, (char*)&RecvSize, SizeSize );
	getsockopt( Socket, SOL_SOCKET, SO_RCVBUF, (char*)&RecvSize, GCC_OPT_INT_CAST &SizeSize );
	setsockopt( Socket, SOL_SOCKET, SO_SNDBUF, (char*)&SendSize, SizeSize );
	getsockopt( Socket, SOL_SOCKET, SO_SNDBUF, (char*)&SendSize, GCC_OPT_INT_CAST &SizeSize );
	debugf( NAME_Init, TEXT("%s: Socket queue %i / %i"), SOCKET_API, RecvSize, SendSize );

	// Bind socket to our port.
	LocalAddr = getlocalbindaddr( *GLog );
	LocalAddr.Port = 0;
	UBOOL HardcodedPort     = 0;
	if( !Connect )
	{
		// Init as a server.
		HardcodedPort = Parse( appCmdLine(), TEXT("PORT="), URL.Port );
		LocalAddr.Port = URL.Port;
	}

	INT AttemptPort = LocalAddr.Port;

	sockaddr_storage LclAddr;
	socklen_t LclAddrLen;
	LocalAddr.GetSockAddr(&LclAddr, &LclAddrLen);
	INT boundport   = bindnextport( Socket, (sockaddr*)&LclAddr, LclAddrLen, HardcodedPort ? 1 : 20, 1 );
	if( boundport==0 )
	{
		Error = FString::Printf( TEXT("%s: binding to port %i failed (%i)"), SOCKET_API, AttemptPort, SocketError() );
		return 0;
	}
	if( !SetNonBlocking( Socket ) )
	{
		Error = FString::Printf( TEXT("%s: SetNonBlocking failed (%i)"), SOCKET_API, SocketError() );
		return 0;
	}

	// Success.
	return 1;
	unguard;
}

UTcpipConnection* UTcpNetDriver::GetServerConnection() 
{
	return (UTcpipConnection*)ServerConnection;
}

//
// Return the NetDriver's socket.  For master server NAT socket opening purposes.
//
FSocketData UTcpNetDriver::GetSocketData()
{
	FSocketData Result;
	Result.Socket = Socket;
	Result.UpdateFromSocket();
	return Result;
}

void UTcpNetDriver::StaticConstructor()
{
	guard(UTcpNetDriver::StaticConstructor);
	new(GetClass(),TEXT("AllowPlayerPortUnreach"),	RF_Public)UBoolProperty (CPP_PROPERTY(AllowPlayerPortUnreach), TEXT("Client"), CPF_Config );
	new(GetClass(),TEXT("LogPortUnreach"),			RF_Public)UBoolProperty (CPP_PROPERTY(LogPortUnreach        ), TEXT("Client"), CPF_Config );
	new(GetClass(),TEXT("MaxConnPerIPPerMinute"),	RF_Public)UIntProperty  (CPP_PROPERTY(MaxConnPerIPPerMinute ), TEXT("Client"), CPF_Config );
	new(GetClass(),TEXT("LogMaxConnPerIPPerMin"),	RF_Public)UBoolProperty (CPP_PROPERTY(LogMaxConnPerIPPerMin ), TEXT("Client"), CPF_Config );
	new(GetClass(),TEXT("LimitConnPerIPRangePerMinute"),	RF_Public)UBoolProperty  (CPP_PROPERTY(LimitConnPerIPRangePerMinute ), TEXT("Client"), CPF_Config );
	unguard;
}

IMPLEMENT_CLASS(UTcpNetDriver);

/*-----------------------------------------------------------------------------
	The End.
-----------------------------------------------------------------------------*/

